/* 

	UART drivers, by Mauro Grassi, 2010.
	
	
*/

#include "uart.h"
#include "mystring.h"
#include "main.h"
#include "hardwarestate.h"
#include "io.h"

#pragma udata uartdata

#pragma code usercode

unsigned char disFix(unsigned char x)
{
	x=x & 0x0F;
	if(x<=9)return x+'0';
	else
	return x+'A'-10;
}

#if 0
void disAUART(unsigned char x)
{
	unsigned char buffer[5];
	
	buffer[0]='0';
	buffer[1]='x';
	buffer[2]=disFix(x>>4);
	buffer[3]=disFix(x);
	buffer[4]='\0';
	putsUART((char*)&buffer[0]);
}
#endif

#if 0
void disWUART(unsigned int x)
{
	unsigned char buffer[8];
	
	buffer[0]='0';
	buffer[1]='x';
	buffer[2]=disFix((unsigned char)(x>>12));
	buffer[3]=disFix((unsigned char)(x>>8));
	buffer[4]=disFix((unsigned char)(x>>4));
	buffer[5]=disFix(x);
	buffer[6]='\0';
	putsUART((char*)&buffer[0]);
}
#endif

#if 0
void disDWUART(unsigned long x)
{
	unsigned char buffer[12];
	
	buffer[0]='0';
	buffer[1]='x';
	buffer[2]=disFix(x>>28);
	buffer[3]=disFix(x>>24);
	buffer[4]=disFix(x>>20);
	buffer[5]=disFix(x>>16);
	buffer[6]=disFix(x>>12);
	buffer[7]=disFix(x>>8);
	buffer[8]=disFix(x>>4);
	buffer[9]=disFix(x);
	buffer[10]='\0';
	putsUART((char*)&buffer[0]);
}
#endif

#if 0
void disDecUART(unsigned long x)
{
	myultoa((DWORD)x, tempString, 4);
	putsUART((char*)tempString);
}
#endif

void putcUART(unsigned char c)
{
	if((RCSTA2bits.SPEN)&&(TXSTA2bits.TXEN))
	{
		TXREG2=c;
		while((RCSTA2bits.SPEN)&&(TXSTA2bits.TXEN)&&(TXSTA2bits.TRMT==0));
	}	
}

void putrsUART(const rom char* instr)
{
	while((*instr)!='\0')
	{
		putcUART(*instr);
		instr++;
	}
}

#if 0
void putsSizeUART(int size, unsigned char* instr)
{
	while((size>0)&&((*instr)!='\0'))
	{
		putcUART(*instr);
		instr++;
		size--;
	}
}
#endif

void putsUART(char* instr)
{
	while((*instr)!='\0')
	{
		putcUART(*instr);
		instr++;
	}
}
	
void powerUART(unsigned char on)
{
	if(on)
	{
		#if(USE_PMDIS)
			PMDIS0bits.UART2MD=0;
		#endif
	}
	else
	{
	
		#if(USE_PMDIS)
			PMDIS0bits.UART2MD=1;
		#endif	
	}
}

void closeUARTState(VIRTUAL_MACHINE* vm, UART_DESCRIPTOR* systemState)
{
	unsigned char txch;
	
	/* closes the hardware resources */
	if(systemState->mode & NO_UART_MODE)
	{
	
	}
	else
	{
		/* wait till all characters have been transmitted! */
		while((RCSTA2bits.SPEN)&&(TXSTA2bits.TXEN)&&(TXSTA2bits.TRMT==0));
		/* close the resources! */
		if(systemState->mode & UART_RX_INT)
		{
			PIE3bits.RC2IE=0;
			PIR3bits.RC2IF=0;
		}
		txch=(systemState->txrxpin>>4) & 0x0F;
		openIOObject(txch, 1);
		RCSTA2=0;
		setRPNValue(txch, 0);
		RPINR16=0x1F;
		powerUART(0);
		clearUARTDescriptor(systemState);
		
	}
}

unsigned char receivedNMEAUART(VIRTUAL_MACHINE* vm)
{
	return (unsigned char)(vm->GPS.GPSPipeState==GPS_ALL_DONE_STATE);
}

void UARTRxPipePut(VIRTUAL_MACHINE* vm, unsigned char g)
{
	unsigned char x;
	
	x=vm->GPS.GPSPipePutPtr;
	if(x<(MAX_GPS_PIPE_LENGTH-1))x++; else x=0;

	if(x==vm->GPS.GPSPipeGetPtr)
	{
		(void)UARTRxPipeGet(vm);
		vm->GPS.GPSError|=GPS_OVERFLOW_ERROR;
	}
	else
	{
		vm->GPS.GPSRxCRC^=g;
		/* Make Room If Possible */	
		/* Adds another char */
		vm->GPS.GPSPipe[vm->GPS.GPSPipePutPtr]=g;
		vm->GPS.GPSPipePutPtr=x;
	}
}

unsigned char UARTRxPipeGet(VIRTUAL_MACHINE* vm)
{
	unsigned char g;
	
	if(vm->GPS.GPSPipeGetPtr!=vm->GPS.GPSPipePutPtr)
	{
		g=vm->GPS.GPSPipe[vm->GPS.GPSPipeGetPtr];
		if(vm->GPS.GPSPipeGetPtr<(MAX_GPS_PIPE_LENGTH-1))vm->GPS.GPSPipeGetPtr++; else 
		vm->GPS.GPSPipeGetPtr=0;
	}
	else
	{
		/* nothing to get */
		vm->GPS.GPSError|=GPS_UNDERFLOW_ERROR;
		g='\0';
	}
	return g;
}

void clearUARTRx(VIRTUAL_MACHINE* vm)
{
	vm->GPS.GPSError=GPS_NO_ERROR;
	vm->GPS.GPSPipePutPtr=0;
	vm->GPS.GPSPipeGetPtr=0;
	vm->GPS.GPSRxCRC=0;
	vm->GPS.GPSLastRx='\0';
	vm->GPS.GPSPipeState=GPS_IDLE_STATE;
	vm->GPS.GPSNewRx=0;
}

unsigned char getUARTError(VIRTUAL_MACHINE* vm)
{
	unsigned char c;
	c=vm->GPS.GPSError;
	vm->GPS.GPSError=GPS_NO_ERROR;
	return c;
}

void initUARTState(VIRTUAL_MACHINE* vm, UART_DESCRIPTOR* systemState)
{
	/* Initialize the Physical UART2 port to match the Descriptor */
	unsigned short baudRate;
	unsigned char  txch;
	unsigned char  rxch;
	
	if(systemState->mode & NO_UART_MODE)
	{
		/* turn it off */
		powerUART(0);
	}
	else
	{
		/* turn on the UART */
		txch=(systemState->txrxpin>>4) & 0x0F;
		rxch=(systemState->txrxpin) & 0x0F;
		openIOObject(txch, 1);
		openIOObject(rxch, 1);
		powerUART(1);
		RCSTA2=0;
		TXSTA2=0;
		BAUDCON2=0;
		if(systemState->mode & UART_OD)
		{
			ODCON2bits.U2OD=1;
		}
		else
		{
			ODCON2bits.U2OD=0;
		}
		if(systemState->mode & UART_NO_RX_INV)
		{
			BAUDCON2bits.RXDTP=0;	
		}
		else
		{
			BAUDCON2bits.RXDTP=1;
		}
		if(systemState->mode & UART_NO_TX_INV)
		{
			BAUDCON2bits.TXCKP=0;
		}
		else
		{
			BAUDCON2bits.TXCKP=1;
		}

		BAUDCON2bits.BRG16=1;
		TXSTA2bits.BRGH=1;
		if(systemState->baudRate>=(LOWEST_BAUD_RATE/BAUD_RATE_FACTOR))
		{
			baudRate=(unsigned short)((float)GetInstructionClock()/((float)systemState->baudRate*BAUD_RATE_FACTOR))-1;
		}
		else
		{
			baudRate=LOWEST_BAUD_REGISTER_VALUE;
		}
		SPBRGH2=(unsigned char)(baudRate>>8);
		SPBRG2=(unsigned char)baudRate;		

		IPR3bits.RC2IP=0;
		PIR3bits.RC2IF=0;
		
		if(systemState->mode & UART_RX_INT)
		{
		// Disabled for now, M.G.
		//	PIE3bits.RC2IE=1;
		}
		else
		{
			PIE3bits.RC2IE=0;
		}
		
		if(systemState->mode & UART_NO_TX)
		{
			TXSTA2bits.TXEN=0;		
		}
		else
		{
			TXSTA2bits.TXEN=1;
		}
		
		if(systemState->mode & UART_NO_RX)
		{
			RCSTA2bits.CREN=0;	
		}
		else
		{
			RCSTA2bits.CREN=1;		
		}
		RCSTA2bits.SPEN=1;
		setRPNValue(txch, RPOR_VALUE_FOR_TX_UART);
		RPINR16=getRPNNumber(rxch);
		openIOObject(txch, 0);
	}
}
